/*******************************************************************************
*  skMPL115A2 - 大気圧センサMPL115A2(I2C)用関数ライブラリ                      *
*                                                                              *
*    skMPL115A2      - MPL115A2関数ライブラリを生成する時の初期化処理          *
*    CoefficientRead - メモリーマップから係数を読み出す処理                    *
*    PressureRead    - メモリーマップから圧力と温度のA/D変換値を読み出す処理   *
*    PressureAVE     - 圧力･温度のA/D変換値を平均化させ気圧値を計算する処理    *
*    PressureCalc    - 気圧値(hPa)を計算する処理                               *
*    AltitudeCalc    - 気圧値(hPa)から高度を計算する処理                       *
*                                                                              *
* ============================================================================ *
*   VERSION  DATE        BY             CHANGE/COMMENT                         *
* ---------------------------------------------------------------------------- *
*   1.00     2013-01-05  きむ茶工房     Create                                 *
* ============================================================================ *
*   Arduino IDE 1.0.1 (Ardino Duemilanove 328)                                 *
*******************************************************************************/
#include <Wire.h>
#include <Arduino.h>
#include "skMPL115A2.h"


float skMPL115A2::a0 ;                  // ａ０係数のデータを保存する変数
float skMPL115A2::b1 ;                  // ｂ１係数のデータを保存する変数
float skMPL115A2::b2 ;                  // ｂ２係数のデータを保存する変数
float skMPL115A2::c12 ;                 // ｃ１２係数のデータを保存する変数
unsigned long skMPL115A2::Press ;       // 圧力の変換値を保存する変数
unsigned long skMPL115A2::Temp ;        // 温度の変換値を保存する変数
int skMPL115A2::Sensor_adrs ;           // I2Cアドレスを保存する変数

/*******************************************************************************
*  skMPL115A2(address)                                                         *
*    MPL115A2関数ライブラリを生成する時の初期化処理(コンストラクタ)            *
*    address : スレーブ(MPL115A2)のI2Cアドレスを指定します                     *
*******************************************************************************/
skMPL115A2::skMPL115A2(int address)
{
     // Ｉ２Ｃの初期化
     Wire.begin() ;                       // マスターとする
     Sensor_adrs = address ;
}
/*******************************************************************************
*  ans = CoefficientRead()                                                     *
*  メモリーマップから係数を読み出す処理                                        *
*    ans  : 戻り値、0=正常終了　それ以外I2C通信エラー下記                      *
*                   1=送ろうとしたデータが送信バッファのサイズを超えた         *
*                   2=スレーブ・アドレスを送信し、NACKを受信した               *
*                   3=データ・バイトを送信し、NACKを受信した                   *
*                   4=その他のエラー                                           *
*                   5=データ受信エラー                                         *
*******************************************************************************/
int skMPL115A2::CoefficientRead()
{
     int ans ;
     unsigned int h , l ;

     Wire.beginTransmission(Sensor_adrs) ;        // 通信の開始
     Wire.write(0x04) ;                           // 係数の読出しコマンド発行
     ans = Wire.endTransmission() ;               // データの送信と通信の終了
     if (ans == 0) {
          ans = Wire.requestFrom(Sensor_adrs,8) ; // 係数データの受信を行う
          if (ans == 8) {
               // ａ０の係数を得る
               h = Wire.read() ;
               l = Wire.read() ;
               a0 = (h << 5) + (l >> 3) + (l & 0x07) / 8.0 ;
               // ｂ１の係数を得る
               h = Wire.read() ;
               l = Wire.read() ;
               b1 = ( ( ( (h & 0x1F) * 0x100 ) + l ) / 8192.0 ) - 3 ;
               // ｂ２の係数を得る
               h = Wire.read() ;
               l = Wire.read() ;
               b2 = ( ( ( ( h - 0x80) << 8 ) + l ) / 16384.0 ) - 2 ;
               // Ｃ１２の係数を得る
               h = Wire.read() ;
               l = Wire.read() ;
               c12 = ( ( ( h * 0x100 ) + l ) / 16777216.0 )  ;
               ans = 0 ;
          } else ans = 5 ;
     }

     return ans ;
}
/*******************************************************************************
*  ans = PressureRead()                                                        *
*  メモリーマップから圧力および温度のＡ／Ｄ変換値を読み出す処理                *
*    ans  : 戻り値、0=正常終了　それ以外I2C通信エラー下記                      *
*                   1=送ろうとしたデータが送信バッファのサイズを超えた         *
*                   2=スレーブ・アドレスを送信し、NACKを受信した               *
*                   3=データ・バイトを送信し、NACKを受信した                   *
*                   4=その他のエラー                                           *
*                   5=データ受信エラー                                         *
*******************************************************************************/
int skMPL115A2::PressureRead()
{
     int ans ;
     unsigned int h , l ;

     // 圧力および温度の変換を開始させる処理
     Wire.beginTransmission(Sensor_adrs) ;        // 通信の開始
     Wire.write(0x12) ;                           // 圧力・温度の変換開始コマンド発行
     Wire.write(0x01) ;
     ans = Wire.endTransmission() ;               // データの送信と通信の終了
     if (ans != 0) return ans ;
     delay(3) ;                                   // ３ｍｓ待つ

     // Ａ／Ｄ変換値を得る処理
     Wire.beginTransmission(Sensor_adrs) ;        // 通信の開始
     Wire.write(0x00) ;                           // 圧力のHighバイトから読込むコマンド発行
     ans = Wire.endTransmission() ;               // データの送信と通信の終了
     if (ans == 0) {
          ans = Wire.requestFrom(Sensor_adrs,4) ; // Ａ／Ｄ変換値データの受信を行う
          if (ans == 4) {
               // 圧力のＡ／Ｄ変換値を得る
               h = Wire.read() ;
               l = Wire.read() ;
               Press = ( ( h * 256 ) + l ) / 64 ;
               // 温度のＡ／Ｄ変換値を得る
               h = Wire.read() ;
               l = Wire.read() ;
               Temp = ( ( h * 256 ) + l ) / 64 ;
               ans = 0 ;
          } else ans = 5 ;
     }

     return ans ;
}
/*******************************************************************************
*  ans = PressureAVE(count)                                                    *
*    圧力･温度のA/D変換値を平均化させ気圧値を計算する処理                      *
*    count : A/D変換値を平均の為に何回読み込むか指定する。                     *
*    ans   : 大気圧値を hPa で返す。                                           *
*******************************************************************************/
float skMPL115A2::PressureAVE(int count)
{
     int i ;
     float ans ;
     unsigned long p , t ;

     p = t = 0 ;
     for (i=0 ; i < count ; i++) {      // 指定回数ぶん読み込む
          PressureRead() ;              // メモリーマップから圧力と温度のA/D変換値を読み出す
          p = p + Press ;
          t = t + Temp ;
     }
     Press = p / count ;                // 単純平均化を行う
     Temp  = t / count ;
     ans = PressureCalc() ;             // 気圧値の計算を行う

     return ans ;
}
/*******************************************************************************
*  ans = PressureCalc()                                                        *
*    気圧値(hPa)を計算する処理                                                 *
*    ans : 大気圧値を hPa で返す。                                             *
*******************************************************************************/
float skMPL115A2::PressureCalc()
{
     float ret , f ;

     f = a0 + ( b1 + c12 * Temp ) * Press + b2 * Temp ;
     ret = f * ( 650.0 / 1023.0 ) + 500.0 ;
     return ret ;
}
/*******************************************************************************
*  ans = AltitudeCalc(pressure,Difference)                                     *
*  気圧値(hPa)から高度を計算する処理                                           *
*    pressure   : 計算した大気圧値を hPa でセットする。                        *
*    Difference : 標高の差を指定する                                           *
*    ans        : 高度値を ｍ で返す。                                         *
*******************************************************************************/
float skMPL115A2::AltitudeCalc(float pressure,int Difference)
{
     float h ;

     h = 44330.8 * (1.0 - pow( (pressure/1013.25) ,  0.190263 )) ;
     h = h + Difference ;
     return h ;
}
